#include "sc7705.h"
#include "panels.h"

static void LCD_power_on(__u32 sel);
static void LCD_power_off(__u32 sel);
static void LCD_bl_open(__u32 sel);
static void LCD_bl_close(__u32 sel);

static void LCD_panel_init(__u32 sel);
static void LCD_panel_exit(__u32 sel);

static void LCD_cfg_panel_info(panel_extend_para *info)
{
	__u32 i = 0, j = 0;
	__u32 items;
	__u8 lcd_gamma_tbl[][2] = {
		//{input value, corrected value}
		{0, 0},
		{15, 15},
		{30, 30},
		{45, 45},
		{60, 60},
		{75, 75},
		{90, 90},
		{105, 105},
		{120, 120},
		{135, 135},
		{150, 150},
		{165, 165},
		{180, 180},
		{195, 195},
		{210, 210},
		{225, 225},
		{240, 240},
		{255, 255},
	};

	__u32 lcd_cmap_tbl[2][3][4] = {
		{
			{LCD_CMAP_G0, LCD_CMAP_B1, LCD_CMAP_G2, LCD_CMAP_B3},
			{LCD_CMAP_B0, LCD_CMAP_R1, LCD_CMAP_B2, LCD_CMAP_R3},
			{LCD_CMAP_R0, LCD_CMAP_G1, LCD_CMAP_R2, LCD_CMAP_G3},
		},
		{
			{LCD_CMAP_B3, LCD_CMAP_G2, LCD_CMAP_B1, LCD_CMAP_G0},
			{LCD_CMAP_R3, LCD_CMAP_B2, LCD_CMAP_R1, LCD_CMAP_B0},
			{LCD_CMAP_G3, LCD_CMAP_R2, LCD_CMAP_G1, LCD_CMAP_R0},
		},
	};


	items = sizeof(lcd_gamma_tbl) / 2;
	for (i = 0; i < items - 1; i++) {
		__u32 num = lcd_gamma_tbl[i + 1][0] - lcd_gamma_tbl[i][0];

		for (j = 0; j < num; j++) {
			__u32 value = 0;

			value = lcd_gamma_tbl[i][1] + ((lcd_gamma_tbl[i + 1][1] - lcd_gamma_tbl[i][1]) * j) / num;
			info->lcd_gamma_tbl[lcd_gamma_tbl[i][0] + j] = (value << 16) + (value << 8) + value;
		}
	}
	info->lcd_gamma_tbl[255] = (lcd_gamma_tbl[items - 1][1] << 16) + (lcd_gamma_tbl[items - 1][1] << 8) + lcd_gamma_tbl[items - 1][1];

	memcpy(info->lcd_cmap_tbl, lcd_cmap_tbl, sizeof(lcd_cmap_tbl));

}

static __s32 LCD_open_flow(__u32 sel)
{
	LCD_OPEN_FUNC(sel, LCD_power_on, 30);		// open lcd power, than delay 30ms
	LCD_OPEN_FUNC(sel, LCD_panel_init, 10);		// open lcd power, than delay 10ms
	LCD_OPEN_FUNC(sel, sunxi_lcd_tcon_enable, 200);	// open lcd controller, than delay 200ms
	LCD_OPEN_FUNC(sel, LCD_bl_open, 0);		// open lcd backlight, than delay 0ms

	return 0;
}

static __s32 LCD_close_flow(__u32 sel)
{
	LCD_CLOSE_FUNC(sel, LCD_bl_close, 200);		// close lcd backlight, than delay 200ms
	LCD_CLOSE_FUNC(sel, sunxi_lcd_tcon_disable, 5);	// close lcd controller, than delay 5ms
	LCD_CLOSE_FUNC(sel, LCD_panel_exit, 20);	// close lcd power, than delay 20ms
	LCD_CLOSE_FUNC(sel, LCD_power_off, 50);		// close lcd power, than delay 50ms

	return 0;
}

static void LCD_power_on(__u32 sel)
{
	sunxi_lcd_power_enable(sel, 0);
	sunxi_lcd_power_enable(sel, 1);
	sunxi_lcd_power_enable(sel, 2);
}

static void LCD_power_off(__u32 sel)
{
	sunxi_lcd_power_disable(sel, 0);
	sunxi_lcd_power_disable(sel, 1);
	sunxi_lcd_power_disable(sel, 2);
}

static void LCD_bl_open(__u32 sel)
{
	sunxi_lcd_pwm_enable(sel);
	sunxi_lcd_backlight_enable(sel);
}

static void LCD_bl_close(__u32 sel)
{
	sunxi_lcd_backlight_disable(sel);
	sunxi_lcd_pwm_disable(sel);
}

#define REGFLAG_DELAY			0xFFFFFFFE
#define REGFLAG_END_OF_TABLE		0xFFFFFFFF

struct LCM_setting_table {
	__u32 cmd;
	__u32 count;
	__u8 para_list[64];
};

/*
#define Width 768
#define Height 1024

#define VFP 16
#define VBP 16
#define VSA 6

#define HFP 110
#define HBP 110
#define HSA 45
*/

typedef enum {
	IC_SC7705 = 1,
	IC_ILI9881C,
} ic_type_t;

static ic_type_t ic_type = IC_ILI9881C;

#define REG_BA_P1	0x33 // parameter 1 of BAh

static struct LCM_setting_table sc7705_initialization_setting[] = {
	{ 0xB9, 0x03, { 0xF1, 0x12, 0x84 } },
	{ 0xBA, 0x1B, { REG_BA_P1, 0x81, 0x05, 0xF9, 0x0E, 0x0E, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x25, 0x00, 0x91, 0x0A, 0x00, 0x00, 0x00, 0x4F, 0x01, 0x00, 0x00, 0x37 } },
	{ 0xB8, 0x01, { 0xA6 } }, // ECP
	{ 0xB3, 0x08, { 0x00, 0x00, 0x00, 0x00, 0x28, 0x28, 0x28, 0x28 } }, // RGB
	{ 0xC0, 0x09, { 0x73, 0x73, 0x50, 0x50, 0x80, 0x00, 0x08, 0x70, 0x00 } }, // SCR
	{ 0xCC, 0x01, { 0x0B } }, // panel, 0x0B:Forward,0x07:Backward
	{ 0xBF, 0x03, { 0x02, 0x10, 0x82 } }, // ECP
	{ 0xBC, 0x01, { 0x46 } }, // VDC
	{ 0xB4, 0x01, { 0x80 } }, // Panel inversion
//	{ 0xB2, 0x02, { 0x40, 0x08 } }, // RSO 720p
	{ 0xB2, 0x02, { 0x00, 0x09 } }, // RSO 768p
	{ 0xE3, 0x0A, { 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x00, 0x00, 0xC0 } }, // EQ
	{ 0xB1, 0x0A, { 0x43, 0x54, 0x23, 0x2F, 0x2F, 0x33, 0x77, 0x01, 0xDB, 0x4C } }, // POWER
	{ 0xB5, 0x02, { 0x15, 0x15 } }, // POWER
	{ 0xB6, 0x02, { 0x86, 0x86 } }, // VCOM
	{ 0xE9, 0x3F, { 0x02, 0x00, 0x10, 0x04, 0x16, 0x80, 0x81, 0x12, 0x31, 0x23, 0x2F, 0x14, 0x80, 0x81, 0x27, 0x04, 0x00, 0x00, 0x0F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0F, 0x00, 0x00, 0x00, 0x44, 0x00, 0x88, 0xFF, 0x00, 0x22, 0x44, 0x66, 0x88, 0x88, 0x88, 0x55, 0x11, 0x88, 0xFF, 0x11, 0x33, 0x55, 0x77, 0x88, 0x88, 0x88, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 } }, // GIP
	{ 0xEA, 0x3F, { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x11, 0x55, 0xFF, 0x88, 0x77, 0x55, 0x33, 0x11, 0x88, 0x88, 0x88, 0x00, 0x44, 0xFF, 0x88, 0x66, 0x44, 0x22, 0x00, 0x88, 0x88, 0x88, 0x23, 0x00, 0x00, 0x00, 0x01, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x80, 0x81, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 } }, // GIP2
	{ 0xE0, 0x22, { 0x03, 0x10, 0x12, 0x27, 0x2D, 0x37, 0x3C, 0x37, 0x05, 0x09, 0x0C, 0x10, 0x11, 0x10, 0x12, 0x13, 0x19, 0x03, 0x10, 0x12, 0x27, 0x2D, 0x3D, 0x3C, 0x37, 0x05, 0x09, 0x0C, 0x10, 0x11, 0x10, 0x12, 0x13, 0x19 } }, // Gamma 2.2
	{ REGFLAG_END_OF_TABLE, 0x00, { } },
};

static struct LCM_setting_table ili9881c_initialization_setting[] = {
	{ 0xFF, 0x03, { 0x98, 0x81, 0x03 } }, // CMD_Page 3
	// GIP_1
	{ 0x01, 0x01, { 0x00 } },
	{ 0x02, 0x01, { 0x00 } },
	{ 0x03, 0x01, { 0x51 } },
	{ 0x04, 0x01, { 0x11 } },
	{ 0x05, 0x01, { 0x00 } },
	{ 0x06, 0x01, { 0x04 } },
	{ 0x07, 0x01, { 0x02 } },
	{ 0x08, 0x01, { 0x00 } },
	{ 0x09, 0x01, { 0x00 } },
	{ 0x0A, 0x01, { 0x00 } },
	{ 0x0B, 0x01, { 0x00 } },
	{ 0x0C, 0x01, { 0x00 } },
	{ 0x0D, 0x01, { 0x00 } },
	{ 0x0E, 0x01, { 0x00 } },
	{ 0x0F, 0x01, { 0x00 } },
	{ 0x10, 0x01, { 0x00 } },
	{ 0x11, 0x01, { 0x00 } },
	{ 0x12, 0x01, { 0x00 } },
	{ 0x13, 0x01, { 0x00 } },
	{ 0x14, 0x01, { 0x00 } },
	{ 0x15, 0x01, { 0x00 } },
	{ 0x16, 0x01, { 0x00 } },
	{ 0x17, 0x01, { 0x00 } },
	{ 0x18, 0x01, { 0x00 } },
	{ 0x19, 0x01, { 0x00 } },
	{ 0x1A, 0x01, { 0x00 } },
	{ 0x1B, 0x01, { 0x00 } },
	{ 0x1C, 0x01, { 0x00 } },
	{ 0x1D, 0x01, { 0x00 } },
	{ 0x1E, 0x01, { 0x40 } },
	{ 0x1F, 0x01, { 0x80 } },
	{ 0x20, 0x01, { 0x00 } },
	{ 0x21, 0x01, { 0x03 } },
	{ 0x22, 0x01, { 0x00 } },
	{ 0x23, 0x01, { 0x00 } },
	{ 0x24, 0x01, { 0x00 } },
	{ 0x25, 0x01, { 0x00 } },
	{ 0x26, 0x01, { 0x00 } },
	{ 0x27, 0x01, { 0x00 } },
	{ 0x28, 0x01, { 0x33 } },
	{ 0x29, 0x01, { 0x11 } },
	{ 0x2A, 0x01, { 0x00 } },
	{ 0x2B, 0x01, { 0x00 } },
	{ 0x2C, 0x01, { 0x00 } },
	{ 0x2D, 0x01, { 0x00 } },
	{ 0x2E, 0x01, { 0x00 } },
	{ 0x2F, 0x01, { 0x00 } },
	{ 0x30, 0x01, { 0x00 } },
	{ 0x31, 0x01, { 0x00 } },
	{ 0x32, 0x01, { 0x00 } },
	{ 0x33, 0x01, { 0x00 } },
	{ 0x34, 0x01, { 0x00 } },
	{ 0x35, 0x01, { 0x00 } },
	{ 0x36, 0x01, { 0x00 } },
	{ 0x37, 0x01, { 0x00 } },
	{ 0x38, 0x01, { 0x00 } },
	{ 0x39, 0x01, { 0x00 } },
	{ 0x3A, 0x01, { 0x00 } },
	{ 0x3B, 0x01, { 0x00 } },
	{ 0x3C, 0x01, { 0x00 } },
	{ 0x3D, 0x01, { 0x00 } },
	{ 0x3E, 0x01, { 0x00 } },
	{ 0x3F, 0x01, { 0x00 } },
	{ 0x40, 0x01, { 0x00 } },
	{ 0x41, 0x01, { 0x00 } },
	{ 0x42, 0x01, { 0x00 } },
	{ 0x43, 0x01, { 0x00 } },
	{ 0x44, 0x01, { 0x00 } },
	// GIP_2
	{ 0x50, 0x01, { 0x01 } },
	{ 0x51, 0x01, { 0x23 } },
	{ 0x52, 0x01, { 0x45 } },
	{ 0x53, 0x01, { 0x67 } },
	{ 0x54, 0x01, { 0x89 } },
	{ 0x55, 0x01, { 0xAB } },
	{ 0x56, 0x01, { 0x01 } },
	{ 0x57, 0x01, { 0x23 } },
	{ 0x58, 0x01, { 0x45 } },
	{ 0x59, 0x01, { 0x67 } },
	{ 0x5A, 0x01, { 0x89 } },
	{ 0x5B, 0x01, { 0xAB } },
	{ 0x5C, 0x01, { 0xCD } },
	{ 0x5D, 0x01, { 0xEF } },
	// GIP_3
	{ 0x5E, 0x01, { 0x11 } },
	{ 0x5F, 0x01, { 0x08 } },
	{ 0x60, 0x01, { 0x08 } },
	{ 0x61, 0x01, { 0x06 } },
	{ 0x62, 0x01, { 0x06 } },
	{ 0x63, 0x01, { 0x00 } },
	{ 0x64, 0x01, { 0x00 } },
	{ 0x65, 0x01, { 0x01 } },
	{ 0x66, 0x01, { 0x01 } },
	{ 0x67, 0x01, { 0x0C } },
	{ 0x68, 0x01, { 0x0C } },
	{ 0x69, 0x01, { 0x0D } },
	{ 0x6A, 0x01, { 0x0D } },
	{ 0x6B, 0x01, { 0x0E } },
	{ 0x6C, 0x01, { 0x0E } },
	{ 0x6D, 0x01, { 0x0F } },
	{ 0x6E, 0x01, { 0x0F } },
	{ 0x6F, 0x01, { 0x02 } },
	{ 0x70, 0x01, { 0x02 } },
	{ 0x71, 0x01, { 0x02 } },
	{ 0x72, 0x01, { 0x02 } },
	{ 0x73, 0x01, { 0x02 } },
	{ 0x74, 0x01, { 0x02 } },
	{ 0x75, 0x01, { 0x06 } },
	{ 0x76, 0x01, { 0x06 } },
	{ 0x77, 0x01, { 0x08 } },
	{ 0x78, 0x01, { 0x08 } },
	{ 0x79, 0x01, { 0x00 } },
	{ 0x7A, 0x01, { 0x00 } },
	{ 0x7B, 0x01, { 0x01 } },
	{ 0x7C, 0x01, { 0x01 } },
	{ 0x7D, 0x01, { 0x0C } },
	{ 0x7E, 0x01, { 0x0C } },
	{ 0x7F, 0x01, { 0x0D } },
	{ 0x80, 0x01, { 0x0D } },
	{ 0x81, 0x01, { 0x0E } },
	{ 0x82, 0x01, { 0x0E } },
	{ 0x83, 0x01, { 0x0F } },
	{ 0x84, 0x01, { 0x0F } },
	{ 0x85, 0x01, { 0x02 } },
	{ 0x86, 0x01, { 0x02 } },
	{ 0x87, 0x01, { 0x02 } },
	{ 0x88, 0x01, { 0x02 } },
	{ 0x89, 0x01, { 0x02 } },
	{ 0x8A, 0x01, { 0x02 } },
	{ 0xFF, 0x03, { 0x98, 0x81, 0x04 } }, // CMD_Page 4
	{ 0x6E, 0x01, { 0x2B } },
	{ 0x6F, 0x01, { 0x36 } },
	{ 0x8D, 0x01, { 0x1A } },
	{ 0x87, 0x01, { 0xBA } },
	{ 0x35, 0x01, { 0x17 } },
	{ 0x3A, 0x01, { 0xA4 } },
	{ 0x3B, 0x01, { 0x98 } },
	{ 0x38, 0x01, { 0x01 } },
	{ 0x39, 0x01, { 0x00 } },
	{ 0x88, 0x01, { 0x0B } },
	{ 0xFF, 0x03, { 0x98, 0x81, 0x01 } }, // CMD_Page 1
	{ 0x22, 0x01, { 0x0A } },
	{ 0x31, 0x01, { 0x00 } },
	{ 0x53, 0x01, { 0x7B } },
	{ 0x55, 0x01, { 0x87 } },
	{ 0x50, 0x01, { 0x95 } },
	{ 0x51, 0x01, { 0x90 } },
	{ 0x60, 0x01, { 0x2C } },
	{ 0x2E, 0x01, { 0x88 } },
	// Gamma START
	// Pos Register
	{ 0xA0, 0x01, { 0x00 } },
	{ 0xA1, 0x01, { 0x08 } },
	{ 0xA2, 0x01, { 0x0F } },
	{ 0xA3, 0x01, { 0x0E } },
	{ 0xA4, 0x01, { 0x0E } },
	{ 0xA5, 0x01, { 0x1E } },
	{ 0xA6, 0x01, { 0x13 } },
	{ 0xA7, 0x01, { 0x18 } },
	{ 0xA8, 0x01, { 0x3A } },
	{ 0xA9, 0x01, { 0x19 } },
	{ 0xAA, 0x01, { 0x26 } },
	{ 0xAB, 0x01, { 0x3E } },
	{ 0xAC, 0x01, { 0x1E } },
	{ 0xAD, 0x01, { 0x1F } },
	{ 0xAE, 0x01, { 0x53 } },
	{ 0xAF, 0x01, { 0x26 } },
	{ 0xB0, 0x01, { 0x2B } },
	{ 0xB1, 0x01, { 0x48 } },
	{ 0xB2, 0x01, { 0x58 } },
	{ 0xB3, 0x01, { 0x23 } },
	// Neg Register
	{ 0xC0, 0x01, { 0x00 } },
	{ 0xC1, 0x01, { 0x08 } },
	{ 0xC2, 0x01, { 0x0F } },
	{ 0xC3, 0x01, { 0x0E } },
	{ 0xC4, 0x01, { 0x0E } },
	{ 0xC5, 0x01, { 0x1E } },
	{ 0xC6, 0x01, { 0x13 } },
	{ 0xC7, 0x01, { 0x18 } },
	{ 0xC8, 0x01, { 0x3A } },
	{ 0xC9, 0x01, { 0x19 } },
	{ 0xCA, 0x01, { 0x26 } },
	{ 0xCB, 0x01, { 0x3E } },
	{ 0xCC, 0x01, { 0x1E } },
	{ 0xCD, 0x01, { 0x1F } },
	{ 0xCE, 0x01, { 0x53 } },
	{ 0xCF, 0x01, { 0x26 } },
	{ 0xD0, 0x01, { 0x2B } },
	{ 0xD1, 0x01, { 0x48 } },
	{ 0xD2, 0x01, { 0x58 } },
	{ 0xD3, 0x01, { 0x23 } },
	// Gamma END
	{ 0xFF, 0x03, { 0x98, 0x81, 0x00 } }, // CMD_Page 0
	{ REGFLAG_END_OF_TABLE, 0x00, { } },
};

static void LCD_panel_init(__u32 sel)
{
	__u32 i;
	unsigned char buffer[4];
	__u32 rx_num = 0;
	const struct LCM_setting_table *init_setting;

	sunxi_lcd_pin_cfg(sel, 1);
	sunxi_lcd_delay_ms(10);
	sunxi_lcd_gpio_set_value(sel, 0, 1);
	sunxi_lcd_delay_ms(50);
	sunxi_lcd_gpio_set_value(sel, 0, 0);
	sunxi_lcd_delay_ms(10);
	sunxi_lcd_gpio_set_value(sel, 0, 1);
	sunxi_lcd_delay_ms(200);

	dsi_dcs_rd(sel, 0xDA, &buffer[0], &rx_num);
	printf("LCD id reg DAh=%02x (%u)\n", buffer[0], rx_num);
	if (rx_num != 1)
		return;

	if (buffer[0] == 0x48) {
		ic_type = IC_SC7705;
		init_setting = sc7705_initialization_setting;
	} else if (buffer[0] == 0x00) {
		ic_type = IC_ILI9881C;
		init_setting = ili9881c_initialization_setting;
	} else {
		return;
	}

	for (i = 0;; i++) {
		if (init_setting[i].cmd == REGFLAG_END_OF_TABLE)
			break;
		else if (init_setting[i].cmd == REGFLAG_DELAY)
			sunxi_lcd_delay_ms(init_setting[i].count);
		else
			dsi_dcs_wr(0, init_setting[i].cmd, init_setting[i].para_list, init_setting[i].count);
	}

	dsi_dcs_wr(sel, 0x11, 0, 0); // Sleep Out
	sunxi_lcd_delay_ms(150);
	dsi_dcs_wr(sel, 0x29, 0, 0); // Display On
	sunxi_lcd_delay_ms(50);

	sunxi_lcd_dsi_clk_enable(sel);

	return;
}

static void LCD_panel_exit(__u32 sel)
{
	sunxi_lcd_dsi_clk_disable(sel);

	dsi_dcs_wr(sel, 0x28, 0, 0); // Display Off
	sunxi_lcd_delay_ms(50);
	dsi_dcs_wr(sel, 0x10, 0, 0); // Sleep In
	sunxi_lcd_delay_ms(150);

	sunxi_lcd_pin_cfg(sel, 0);

	return ;
}

static __s32 LCD_user_defined_func(__u32 sel, __u32 para1, __u32 para2, __u32 para3)
{
	return 0;
}

__lcd_panel_t sc7705_panel = {
	/* panel driver name, must mach the name of lcd_drv_name in sys_config.fex */
	.name = "sc7705",
	.func = {
		.cfg_panel_info = LCD_cfg_panel_info,
		.cfg_open_flow = LCD_open_flow,
		.cfg_close_flow = LCD_close_flow,
		.lcd_user_defined_func = LCD_user_defined_func,
	},
};
